﻿using System.Threading.Tasks;
using System;
using Hims.Domain.Services;
using Hims.Api.Models;
using Hims.Shared.Library.Enums;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Hims.Api.Utilities;
using Hims.Shared.UserModels;
using System.Collections.Generic;
using Hims.Shared.UserModels.Filters;
using Hims.Shared.DataFilters;
using Hims.Shared.UserModels.Laboratory;
using Hims.Shared.EntityModels;

namespace Hims.Api.Controllers
{
    /// <summary>
    /// The Tpa Controller
    /// </summary>
    [Route("api/tpa")]

    public class TpaController : BaseController
    {
        /// <summary>
        /// The Tpa Service
        /// </summary>
        
        private readonly ITpaService service;
        /// <summary>
        /// The AuditLog Service
        /// </summary>
    
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public TpaController(ITpaService service, IAuditLogService auditLogServices)
        {
            this.service = service;
            this.auditLogServices = auditLogServices;
        }
        /// <summary>
        /// The Add Tpa
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>

        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> AddTpa([FromBody]TpaModel model, [FromHeader] LocationHeader location)
        {
            model = (TpaModel)EmptyFilter.Handler(model);
            int result = await service.AddTpa(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.Tpa,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" {model.CreatedByName} has added Tpa : <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("dd/MM/yyyy, hh:mm tt")}",
                LocationId = Convert.ToInt32(location.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return Ok(result);

        }

        /// <summary>
        /// The Change Status.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("change-status")]
        public async Task<ActionResult> ChangeStatus([FromBody] TpaModel model, [FromHeader] LocationHeader location)
        {
            model = (TpaModel)EmptyFilter.Handler(model);
            int result = await service.ChangeStatus(model);

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Tpa,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" {model.ModifiedByName} has {((bool)model.Active ? "Deactivate" : "Activate")}  Tpa  - <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("dd/MM/yyyy, hh:mm tt")}",
                LocationId = Convert.ToInt32(location.LocationId)
            };

            await this.auditLogServices.LogAsync(auditLogModel);
            return Ok(result);
        }

        /// <summary>
        /// The Get All Tpa.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> GetAllTpa([FromBody] TpaModel model)
        {
            var result = await service.GetAllTpa(model);
            return Ok(result);
        }


        /// <summary>
        /// The Update Tpa.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPut]
        [Route("update")]
        public async Task<ActionResult> UpdateTpa([FromBody] TpaModel model , [FromHeader] LocationHeader location)
        {
            model = (TpaModel)EmptyFilter.Handler(model);
            var result = await service.UpdateTpa(model);

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Tpa,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" {model.ModifiedByName} has updated Tpa : <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("dd/MM/yyyy, hh:mm tt")}",
                LocationId = Convert.ToInt32(location.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return Ok(result);
        }
    }
}
